
/* RGB LED Strip Driver/Controller software for ATmega48
   Copyright 2014 Silicon Chip Publications
   Written by Nicholas Vinen */

#define F_CPU 8000000L
#undef PROTOTYPE // only set this for very early boards which had a slightly different pin mapping
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include <string.h>
#include <stdlib.h>

/* hfuse: 0xdc
   lfuse: 0xc2 */

typedef struct {
  unsigned char when, portd, portc, portb;
} pwm_entry;

static const signed char quarter_sine[64] = { 0, 3, 6, 9, 12, 16, 19, 22, 25, 28, 31, 34, 37, 40, 43, 46, 49, 51, 54, 57, 60, 63, 65, 68, 71, 73, 76, 78, 81, 83, 85, 88, 90, 92, 94, 96, 98, 100, 102, 104, 106, 107, 109, 111, 112, 113, 115, 116, 117, 118, 120, 121, 122, 122, 123, 124, 125, 125, 126, 126, 126, 127, 127, 127 };

unsigned char cur_values[18], last_values[18];
pwm_entry pwm[2][19];
unsigned char num_pwm_entries[2], cur_pwm_entry;
volatile unsigned char ticks, which_pwm, cur_pwm, seconds;
unsigned char pattern, cycling = 1;


typedef struct {
  unsigned short pos;
  unsigned short inten;
  unsigned char fade;
  signed char speed;
  unsigned char rgb[3];
} pattern_line;

#define MAXLINES 4
pattern_line lines[MAXLINES];
unsigned char num_lines;

ISR(WDT_vect) {
}

ISR(TIMER1_COMPA_vect) {
  pwm_entry* ppwm = &pwm[which_pwm][cur_pwm_entry];
  PORTD = ppwm->portd;
  PORTC = ppwm->portc;
#ifndef PROTOTYPE
  PORTB = ppwm->portb | (PORTB & 0xB8);
#else
  PORTB = ppwm->portb | (PORTB & 0x3c);
#endif
  if( ++cur_pwm_entry == num_pwm_entries[which_pwm] ) {
    cur_pwm_entry = 0;
    which_pwm = cur_pwm;
    if( ++ticks == 0 )
      ++seconds;
  }

  ppwm = &pwm[which_pwm][cur_pwm_entry];
  OCR1AH = ppwm->when >> 1;
  OCR1AL = (ppwm->when&1) ? 128 : 0;
}

void turn_off_bit(unsigned char index, pwm_entry* pentry) {
  if( index < 8 )
    pentry->portd &= ~(1<<index);
  else if( index < 14 )
    pentry->portc &= ~(1<<(index-8));
#ifndef PROTOTYPE
  else if( index < 17 )
    pentry->portb &= ~(1<<(index-14));
  else
    pentry->portb &= ~(1<<6);
#else
  else if( index < 16 )
    pentry->portb &= ~(1<<(index-14));
  else
    pentry->portb &= ~(1<<(index-10));
#endif
}

void update_pwm_entries() {
  unsigned char val, i, last, found, new_num;
  pwm_entry* pwms;

  while( which_pwm != cur_pwm )
    ;

  pwms = pwm[cur_pwm^1];
  pwms[0].portd = 0xff; /* outputs  0- 7, PD0-PD7 */
  pwms[0].portc = 0x3f; /* outputs  8-13, PC0-PC5 */
#ifndef PROTOTYPE
  pwms[0].portb = 0x47; /* outputs 14-17, PB0-PB2 + PB6 */
#else
  pwms[0].portb = 0xC3; /* outputs 14-17, PB0-PB1 + PB6-PB7 */
#endif

  for( i = 0; i < 18; ++i ) {
    if( cur_values[i] == 0 )
      turn_off_bit(i, &pwms[0]);
  }

  last = 0;
  for( new_num = 1; new_num < 19; ++new_num ) {
    val = 255;
    found = 0;
    for( i = 0; i < 18; ++i ) {
      if( cur_values[i] <= val && cur_values[i] > last ) {
        val = cur_values[i];
        found = 1;
      }
    }
    if( !found )
      break;
    memcpy(&pwms[new_num], &pwms[new_num-1], sizeof(pwms[new_num]));
    pwms[new_num].when = val;
    for( i = 0; i < 18; ++i ) {
      if( cur_values[i] == val ) {
        turn_off_bit(i, &pwms[new_num]);
      }
    }
    last = val;
  }
  num_pwm_entries[cur_pwm^1] = new_num;
  cur_pwm ^= 1;
}

unsigned short readADC(unsigned char channel) {
  ADCSRA = _BV(ADEN)|_BV(ADPS2)|_BV(ADPS1);
  if( ADMUX != (channel|(1<<REFS0)) ) {
    unsigned char i;
    ADMUX = channel|(1<<REFS0);
    for( i = 0; i < 100; ++i )
      asm volatile ("nop");
  }
  ADCSRA |= (1 << ADIF);
  ADCSRA = (1 << ADEN)|(1 << ADPS2)|(1 << ADPS1)|(1 << ADSC);
  loop_until_bit_is_set(ADCSRA, ADIF);
  return ADCL|(ADCH<<8);
}

void WDT_off() {
  cli();
  asm volatile("wdr");
  /* Clear WDRF in MCUSR */
  MCUSR &= ~(1<<WDRF);
  /* Write logical one to WDCE and WDE */
  /* Keep old prescaler setting to prevent unintentional time-out */
  WDTCSR |= (1<<WDCE) | (1<<WDIE);
  /* Turn off WDT */
  WDTCSR = 0x00;
  sei();
}

void WDT_on() {
  cli();
  asm volatile("wdr");
  /* Start timed equence */
  WDTCSR |= (1<<WDCE) | (1<<WDIE);
  /* Set new prescaler(time-out) value = 128K cycles (~1.0 s) */
  WDTCSR = (1<<WDIE) | (1<<WDP2) | (1<<WDP1);
  sei();
}

void Powerdown() {
  SMCR = _BV(SM1)|_BV(SE); // power-down
  asm volatile("sleep");
  SMCR = 0;
}

signed char sine(unsigned char pos) {
  if( pos < 128 )
    if( pos < 64 )
      return quarter_sine[pos];
    else
      return quarter_sine[127-pos];
  else
    if( pos < 192 )
      return -quarter_sine[pos-128];
    else
      return -quarter_sine[255-pos];
}

unsigned short button1state, button2state;
unsigned char button1pressed, button2pressed;
unsigned char my_ticks[3];

int main(void) {
  unsigned char i, last_ticks;
  unsigned short supply_voltage, brightness_setting, brightness_correction_factor;

  update_pwm_entries();
  which_pwm = 1;
  i = eeprom_read_byte(0);
  cycling = (i&16) ? 1 : 0;
  pattern = (i&15);
  if( pattern > 9 )
    pattern = 0;

  ICR1H = 128;
  ICR1L = 0; // ICR1 = 32768
  OCR1AH = 0;
  OCR1AL = 0;
  TCCR1B = 25; // use system clock, ICR1 CTC mode
  TIMSK1 = 2;  // enable overflow interrupt
  sei();
  TIFR1 |= 2;

  DDRD = 0xff; /* outputs  0- 7, PD0-PD7 */
  DDRC = 0x3f; /* outputs  8-13, PC0-PC5 */
#ifndef PROTOTYPE
  DDRB = 0x47; /* outputs 14-17, PB0-PB2 + PB6 */
  PORTB = (1<<3)|(1<<7); /* enable pull-ups for pushbutton inputs PB3 & PB7 */
#else
  DDRB = 0xC3; /* outputs 14-17, PB0-PB1 + PB6-PB7 */
  PORTB = (1<<3)|(1<<4); /* enable pull-ups for pushbutton inputs PB3 & PB4 */
#endif

  while(1) {
    last_ticks = ticks;
    while( last_ticks == ticks )
      ;
    supply_voltage = 0;
    readADC(7);
    for( i = 0; i < 4; ++i )
      supply_voltage += readADC(7);
    supply_voltage >>= 2;
    /* If 1024 = 20V then 11.5V = 589 but we need to fudge it a bit
        due to the relatively high source impedance. */
    if( supply_voltage < 540 ) {
      do {
        cli();
        TCCR1B = 0;
        TIMSK1 = 0;
        PORTB = 0;
        PORTC = 0;
        PORTD = 0;
        ADCSRA = 0;
        WDT_on();
        Powerdown();
        WDT_off();
        supply_voltage = readADC(7);
        _delay_ms(10);
        supply_voltage = readADC(7);
      } while( supply_voltage < 590 ); // ~12.2V
      TCCR1B = 25;
      TIMSK1 = 2;  // enable overflow interrupt
      sei();
    }
    if( supply_voltage < 570 )
      brightness_correction_factor = 256;
    else
      brightness_correction_factor = 256 - (supply_voltage - 570) / 2;

    readADC(6);
    brightness_setting = 0;
    for( i = 0; i < 4; ++i )
      brightness_setting += readADC(6);
    brightness_correction_factor = ((unsigned long)brightness_correction_factor * (unsigned long)brightness_setting) >> 8;

    button1state = (button1state<<1) | ((PINB>>3)&1);
#ifndef PROTOTYPE
    button2state = (button2state<<1) | ((PINB>>7)&1);
#else
    button2state = (button2state<<1) | ((PINB>>4)&1);
#endif
    if( button1state == 0 && !button1pressed ) {
      button1pressed = 1;
      cycling = button2pressed;
      seconds = 0;
      if( --pattern > 9 )
        pattern = 9;
      num_lines = 0;

      eeprom_write_byte(0, (cycling ? 16 : 0) | pattern);
    } else if( button1state == 0xFFFF ) {
      button1pressed = 0;
    }

    if( button2state == 0 && !button2pressed ) {
      button2pressed = 1;
      cycling = button1pressed;
      seconds = 0;
      if( ++pattern > 9 )
        pattern = 0;
      num_lines = 0;

      eeprom_write_byte(0, (cycling ? 16 : 0) | pattern);
    } else if( button2state == 0xFFFF ) {
      button2pressed = 0;
    }

    if( pattern == 0 ) {
      if( !(last_ticks&3) )
        ++my_ticks[0];
      for( i = 0; i < 18; ++i )
        cur_values[i] = sine( (my_ticks[0] + i * 14 + (i%3) * 85) & 255 ) + 128;
    } else if( pattern == 1 ) {
      if( !(last_ticks&1) )
        ++my_ticks[0];
      if( !(last_ticks%5) )
        ++my_ticks[1];
      if( !(last_ticks%9) )
        ++my_ticks[2];
      for( i = 0; i < 18; ++i )
        cur_values[i] = sine( (my_ticks[i % 3] + i * 14) & 255 ) + 128;
    } else if( pattern == 2 ) {
      if( !(last_ticks&3) )
        ++my_ticks[0];
      for( i = 0; i < 18; ++i ) {
        signed short val = sine( (my_ticks[0] + (i/3) * 14 + (i%3) * 85) & 255 );
        if( val < 0 )
          val = 0;
        cur_values[i] = val;
      }
    } else if( pattern == 3 ) {
      if( !(last_ticks&7) )
        ++my_ticks[0];
      for( i = 0; i < 18; ++i ) {
        signed short val = sine( (my_ticks[0] + (i/3) * 14 + (i%3) * 85) & 255 ) + 64;
        if( val < 0 )
          val = 0;
        cur_values[i] = val;
      }
    } else if( pattern >= 4 && pattern <= 7 ) {
      memset(cur_values, 0, sizeof(cur_values));
      for( i = 0; i < num_lines; ++i ) {
        unsigned char j;
        for( j = 0; j < 6; ++j ) {
          signed short distance = (lines[i].pos >> 3) - ((j + 1) * 0x100);
          if( distance < 0 )
            distance = -distance;
          if( distance < 0x100 ) {
            unsigned short influence = 0x100 - distance, k, val;
            if( pattern > 5 )
              influence = (influence * (unsigned char)(sine(lines[i].inten>>3) + 1)) >> 7;
            for( k = 0; k < 3; ++k ) {
              val = (unsigned short)cur_values[j*3+k] + ((influence * (unsigned short)lines[i].rgb[k])>>8);
              if( val > 255 )
                val = 255;
              cur_values[j*3+k] = val;
            }
          }
        }
      }
      if( pattern <= 5 ) {
        for( i = 0; i < num_lines; ++i ) {
          signed short new_pos = (signed short)lines[i].pos + (signed short)lines[i].speed;
          if( new_pos < 0 || new_pos > (0x6FF<<3) ) {
            memmove(lines+i, lines+i+1, (num_lines-i-1)*sizeof(*lines));
            --num_lines;
            --i;
          } else {
            lines[i].pos = new_pos;
          }
        }
        if( num_lines < MAXLINES && (pattern == 5 || !(rand()&31)) ) {
          signed short val;
          lines[num_lines].pos = ((rand()>>1)&1) ? (0x6FF<<3) : 0;
          lines[num_lines].speed = 8 + (rand()&31);
          if( lines[num_lines].pos > 0 )
            lines[num_lines].speed = -lines[num_lines].speed;
          for( i = 0; i < 3; ++i )
            lines[num_lines].rgb[i] = rand()&255;
          if( pattern == 4 ) {
            val = (rand()&511) - (signed short)lines[num_lines].rgb[0] - (signed short)lines[num_lines].rgb[1];
            if( val < 0 )
              val = 0;
            else if( val > 255 )
              val = 255;
            lines[num_lines].rgb[2] = val;
          } else {
            lines[num_lines].rgb[(rand()>>1)%3] = 0;
          }
          ++num_lines;
        }
      } else {
        for( i = 0; i < num_lines; ++i ) {
          signed short new_pos = lines[i].pos + lines[i].speed;
          lines[i].inten += lines[i].fade;
          if( new_pos < 0 || new_pos > (0x6FF<<3) || lines[i].inten > (127<<3) ) {
            memmove(lines+i, lines+i+1, (num_lines-i-1)*sizeof(*lines));
            --num_lines;
            --i;
          } else {
            lines[i].pos = new_pos;
          }
        }
        if( num_lines < MAXLINES && !(rand()&3) ) {
          signed short val;
          lines[num_lines].pos = (((rand()>>3)%6 + 1) * 0x100)<<3;
          lines[num_lines].inten = 0;
          if( pattern == 7 ) {
            lines[num_lines].speed = 4 + (rand()&31);
            if( lines[num_lines].pos > 0x600 || (lines[num_lines].pos > 0x100 && ((rand()>>1)&1)) )
              lines[num_lines].speed = -lines[num_lines].speed;
          } else {
            lines[num_lines].speed = 0;
          }
          lines[num_lines].fade = (rand()&15)+2;
          for( i = 0; i < 2; ++i )
            lines[num_lines].rgb[i] = rand()&255;
          val = (rand()&511) - (signed short)lines[num_lines].rgb[0] - (signed short)lines[num_lines].rgb[1];
          if( val < 0 )
            val = 0;
          else if( val > 255 )
            val = 255;
          lines[num_lines].rgb[2] = val;
          ++num_lines;
        }
      }
    } else if( pattern >= 8 ) {
      memcpy(cur_values, last_values, sizeof(cur_values));
      for( i = 0; i < 18; ++i )
        if( cur_values[i] > 0 )
          --cur_values[i];
      if( !((rand()>>1)&15) ) {
        unsigned char pos;
        unsigned short val;
        pos = (rand()>>1)%18;
        val = cur_values[pos] + rand()%192 + 63;
        if( val > 255 )
          val = 255;
        cur_values[pos] = val;
      }
      if( pattern == 9 && !(last_ticks&7) ) {
        for( i = 0; i < 18; ++i ) {
          unsigned short val = cur_values[i];
          if( val > 0 )
            --val;
          if( i > 0 )
            val += last_values[i-1] >> 6;
          if( i < 17 )
            val += last_values[i+1] >> 6;
          if( val > 255 )
            val = 255;
          cur_values[i] = val;
        }
      }
      memcpy(last_values, cur_values, sizeof(last_values));
    }

    for( i = 0; i < 18; ++i )
      cur_values[i] = ((unsigned long)cur_values[i] * (unsigned long)brightness_correction_factor) >> (i%3 == 1 ? 13 : 12);
    update_pwm_entries();

    if( cycling && seconds >= 60 ) {
      if( ++pattern > 9 )
        pattern = 0;
      seconds = 0;
      num_lines = 0;
    }
  }

  return 0;
}
